<?php
namespace PowerElements;

use \Elementor\Repeater;
use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Scheme_Typography;
use \Elementor\Icons_Manager;


	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	 
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class PowerElements_Social extends Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_elements_social';
		}
		
		/**
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Social', 'power-elements' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon. 
		 */
		public function get_icon() {
			return 'eicon-social-icons';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_social',
				[
					'label' => __( 'Social', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_social_view',
				[
					'label'   => __( 'View', 'power-elements' ),
					'type'    => Controls_Manager::HIDDEN,
					'default' => 'traditional',
				]
			);
			$this->add_control(
				'peeao_social_style',
				[
					'label'   => __( 'Type', 'power-elements' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'only_icon',
					'options' => [
						'only_icon'   => __( 'Show Only Icon', 'power-elements' ),
						'only_text'   => __( 'Show Only Text', 'power-elements' ),
						'both' => __( 'Show Both', 'power-elements' ),
					],
				]
			);
			$repeater = new Repeater();
			$repeater->add_control(
				'text',
				[
					'label'       => __( 'Text', 'power-elements' ),
					'type'        => Controls_Manager::TEXT,
					'default'     => __( 'Facebook', 'power-elements' ),
					'placeholder' => __( 'Facebook', 'power-elements' ),
				]
			);
			$repeater->add_control(
				'link',
				[
					'label'       => __( 'Link', 'power-elements' ),
					'type'        => Controls_Manager::URL,
					'placeholder' => __( 'https://your-link.com', 'power-elements' ),
					'default'     => [
						'url' => '#',
					],
				]
			);
			$repeater->add_control(
				'icons',
				[
					'label' => esc_html__( 'Icon', 'power-elements' ),
					'type' => Controls_Manager::ICONS,
					'fa4compatibility' => 'icon',
					'default' => [
						'value' => '',
						'library' => 'peeaoicons',
					],
				]
			);
			$repeater->add_control(
				'icons_color',
				[
					'label' => __( 'Color', 'power-elements' ),
					'type' => \Elementor\Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} {{CURRENT_ITEM}} a i' => 'color: {{VALUE}}'
					],
				]
			);
			$this->add_control(
				'peeao_social_list',
				[
					'label'       => '',
					'type'        => Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'text' => __( 'Facebook', 'power-elements' ),
							'icon' => 'fa fa-facebook',
						],
						[
							'text' => __( 'Twitter', 'power-elements' ),
							'icon' => 'fa fa-twitter',
						],
						[
							'text' => __( 'Instagram', 'power-elements' ),
							'icon' => 'fa fa-instagram',
						],
					],
					'title_field' => '<i class="{{ icon }}" aria-hidden="true"></i>{{{ text }}}',
				]
			);
			
			$this->add_responsive_control(
				'peeao_social_align',
				[
					'label'        => __('Alignment', 'power-elements'),
					'type'         => Controls_Manager::CHOOSE,
					'options'      => [
						'left'    => [
							'title' => __('Left', 'power-elements'),
							'icon'  => 'fa fa-align-left',
						],
						'center'  => [
							'title' => __('Center', 'power-elements'),
							'icon'  => 'fa fa-align-center',
						],
						'right'   => [
							'title' => __('Right', 'power-elements'),
							'icon'  => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __('Justified', 'power-elements'),
							'icon'  => 'fa fa-align-justify',
						],
					],
					'prefix_class' => 'elementor%s-align-',
					'default'      => '',
				]
			);
		
			$this->end_controls_section();
			
			$this->start_controls_section(
				'peeao_social_item_wrapper_section_tab',
				[
					'label' => __( 'Item Wrapper', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'tabs_social_item_wrapper'
			);
	
			$this->start_controls_tab(
				'tabs_social_item_wrapper_normal',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_social_icon_background_color',
				[
					'label'     => __( 'BG Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}}  a.power_social_icon' => 'background-color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'      => 'peeao_social_item_wrapper_border',
					'selector'  => '{{WRAPPER}} .power_social_icon',
					'separator' => 'before',
				]
			);
			$this->add_responsive_control(
				'peeao_social_item_wrapper_border_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_social_item_wrapper_box_shadow',
					'selector' => '{{WRAPPER}} .power_social_icon',
				]
			);
			$this->end_controls_tab();
	
			$this->start_controls_tab(
				'tabs_social_item_wrapper_hover',
				[
					'label' => __( 'Hover', 'power-elements' ),
				]
			);
	
			$this->add_control(
				'peeao_social_icon_background_color_hover',
				[
					'label'     => __( 'BG Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}}  a.power_social_icon:hover' => 'background-color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'      => 'peeao_social_item_wrapper_border_hover',
					'selector'  => '{{WRAPPER}} a.power_social_icon:hover',
					'separator' => 'before',
				]
			);
			
			$this->add_responsive_control(
				'peeao_social_item_wrapper_hover_border_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_social_item_wrapper_hover_social_box_shadow',
					'selector' => '{{WRAPPER}} a.power_social_icon:hover',
				]
			);
	
			$this->end_controls_tab();
			$this->end_controls_tabs();


			$this->add_responsive_control(
				'peeao_social_item_wrapper_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->add_responsive_control(
				'peeao_social_item_wrapper_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->end_controls_section();
			
			$this->start_controls_section(
				'peeao_social_icon_section_tab',
				[
					'label' => __( 'Social Icon', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->start_controls_tabs(
				'tabs_social_icon'
			);
			$this->start_controls_tab(
				'tabs_social_icon_normal',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_social_icon_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon i' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'      => 'peeao_social_icon_border',
					'selector'  => '{{WRAPPER}} a.power_social_icon .power_social_icon_class i',
					'separator' => 'before',
				]
			);
			$this->add_responsive_control(
				'peeao_social_icon_border_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon .power_social_icon_class i' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_social_icon_box_shadow',
					'selector' => '{{WRAPPER}} a.power_social_icon .power_social_icon_class i',
				]
			);
			$this->end_controls_tab();
	
			$this->start_controls_tab(
				'tabs_social_icon_hover',
				[
					'label' => __( 'Hover', 'power-elements' ),
				]
			);
	
			$this->add_control(
				'peeao_social_icon_hover_color',
				[
					'label'     => __( 'Hover Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon:hover i' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_control(
				'peeao_social_icon_background_hover_color',
				[
					'label'     => __( 'BG Hover Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon:hover .power_social_icon_class i' => 'background-color: {{VALUE}};',
						'{{WRAPPER}} a.power_social_icon:focus .power_social_icon_class i' => 'background-color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'      => 'peeao_social_icon_border_hover',
					'selector'  => '{{WRAPPER}} a.power_social_icon:hover .power_social_icon_class i, {{WRAPPER}} a.power_social_icon:focus .power_social_icon_class i',
					'separator' => 'before',
				]
			);
			$this->add_responsive_control(
				'peeao_social_icon_border_radius_hover',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon:hover .power_social_icon_class i' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						'{{WRAPPER}} a.power_social_icon:focus .power_social_icon_class i' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_social_icon_box_shadow_hover',
					'selector' => '{{WRAPPER}} a.power_social_icon:hover .power_social_icon_class i, {{WRAPPER}} a.power_social_icon:focus .power_social_icon_class i',
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();


			$this->add_control(
				'peeao_social_icon_size',
				[
					'label'     => __( 'Icon Size', 'power-elements' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [ 'px', 'em' ],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 60,
							'step' => 1,
						],
						'em'  => [
							'min' => 0,
							'max' => 60,
							'step' => 1,
						],
					],
					'default'    => [
						'unit' => 'px',
						'size' => 24,
					],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon .power_social_icon_class i' => 'font-size: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'peeao_social_icon_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon .power_social_icon_class i' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->add_responsive_control(
				'peeao_social_icon_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon .power_social_icon_class i' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'peeao_social_text_section_tab',
				[
					'label' => __( 'Social Text', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'peeao_social_text_typography',
					'label'    => __( 'Typography', 'power-elements' ),
					'scheme'   => Scheme_Typography::TYPOGRAPHY_1,
					'selector' => '{{WRAPPER}} a.power_social_icon .power_social_icon_name',
				]
			);

			$this->start_controls_tabs(
				'tabs_social_text'
			);
			$this->start_controls_tab(
				'tabs_social_text_normal',
				[
					'label' => __( 'Normal', 'power-elements' ),
				]
			);
			$this->add_control(
				'peeao_social_text_color',
				[
					'label'     => __( 'Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon .power_social_icon_name' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'      => 'peeao_social_text_border',
					'selector'  => '{{WRAPPER}} a.power_social_icon .power_social_icon_name, {{WRAPPER}} a.power_social_icon:focus .power_social_icon_class i',
					'separator' => 'before',
				]
			);
			$this->add_responsive_control(
				'peeao_social_text_border_radius',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon .power_social_icon_name' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						'{{WRAPPER}} a.power_social_icon .power_social_icon_name' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_social_text_box_shadow',
					'selector' => '{{WRAPPER}} a.power_social_icon .power_social_icon_name, {{WRAPPER}} a.power_social_icon:focus .power_social_icon_name',
				]
			);
			$this->end_controls_tab();
	
			$this->start_controls_tab(
				'tabs_social_text_hover',
				[
					'label' => __( 'Hover', 'power-elements' ),
				]
			);
	
			$this->add_control(
				'peeao_social_hover_text_color',
				[
					'label'     => __( 'Hover Color', 'power-elements' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} a.power_social_icon:hover .power_social_icon_name' => 'color: {{VALUE}};',
						'{{WRAPPER}} a.power_social_icon:focus .power_social_icon_name' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'      => 'peeao_social_text_border_hover',
					'selector'  => '{{WRAPPER}} a.power_social_icon:hover .power_social_icon_name, {{WRAPPER}} a.power_social_icon:focus .power_social_icon_class i',
					'separator' => 'before',
				]
			);
			$this->add_responsive_control(
				'peeao_social_text_border_radius_hover',
				[
					'label'      => __( 'Border Radius', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon:hover .power_social_icon_name' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						'{{WRAPPER}} a.power_social_icon:focus .power_social_icon_name' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'peeao_social_text_box_shadow_hover',
					'selector' => '{{WRAPPER}} a.power_social_icon:hover .power_social_icon_name, {{WRAPPER}} a.power_social_icon:focus .power_social_icon_class i',
				]
			);
	
			$this->end_controls_tab();
			$this->end_controls_tabs();




			$this->add_responsive_control(
				'peeao_social_text_padding',
				[
					'label'      => __( 'Padding', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon .power_social_icon_name' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->add_responsive_control(
				'peeao_social_text_margin',
				[
					'label'      => __( 'Margin', 'power-elements' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} a.power_social_icon .power_social_icon_name' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);
			$this->end_controls_section();
		}
		
		
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings();
			extract($settings);
			
			if( $peeao_social_style == 'only_text'){
				$class = 'power_social_icon_onlytext';
			}elseif($peeao_social_style == 'both'){
				$class = 'power_social_icon_showtext';
			}elseif($peeao_social_style == 'only_icon'){
				$class = 'power_social_icon_hidetext';
			}

			$this->add_render_attribute( 'wrapper', 'class', [
				'power_social_icon_wrapper',
				'power_social_icon_rounded',
				$class,
			] );
			?>

			<div class="peeao-wrapper"><ul <?php echo $this->get_render_attribute_string( 'wrapper' ); ?>>
			<?php foreach ( $peeao_social_list as $item ) {
		 
				// new icon
				$migrated = isset( $item['__fa4_migrated']['icons'] );
				// Check if its a new widget without previously selected icon using the old Icon control
				$is_new = empty( $item['icon'] ); ?>
				
				<li class = "elementor-repeater-item-<?php echo $item['_id']; ?>"><a href="<?php echo $item['link']['url']; ?>" class="power_social_icon">
					<?php if($peeao_social_style != 'only_text'){
						?><span class="power_social_icon_class">
							
						<?php if ( $is_new || $migrated ) {
							Icons_Manager::render_icon( $item['icons'], [ 'aria-hidden' => 'true' ] );
						} else {
							?>
							<i class="<?php echo esc_attr($item['icon']); ?>" aria-hidden="true"></i>
							<?php
						} ?>
						</span><?php
					} 
					
					if($peeao_social_style != 'only_icon'){ ?>
						<span class="power_social_icon_name"><?php echo esc_attr( $item['text'] ); ?></span>
					<?php }
					?>
				</a></li>
				

			<?php } ?>
			</ul></div>

		<?php }
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {}
	}