<?php
namespace PowerElements;

use \Elementor\Widget_Base;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Controls_Manager;


	if ( ! defined( 'ABSPATH' ) ) {
		exit; // Exit if accessed directly.
	}
	
	/**
	 * Power Element social widget.
	 *
	 * Elementor widget that displays a social with the ability to control every
	 * aspect of the social design.
	 *
	 * @since 1.0.0
	 */
	class PowerElements_Logo extends Widget_Base {
		
		/**
		 * Get widget name.
		 *
		 * Retrieve social widget name.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'power_elements_logo';
		}
		
		/** 
		 * Get widget title.
		 *
		 * Retrieve social widget title.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return __( 'Logo', 'power-elements' );
		}
		
		/**
		 * Get widget icon.
		 *
		 * Retrieve social widget icon.
		 *
		 * @since 1.0.0
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-logo';
		}
		
		/**
		 * Get widget categories.
		 *
		 * Retrieve the list of categories the social widget belongs to.
		 *
		 * Used to determine where to display the widget in the editor.
		 *
		 * @since 2.0.0
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [ 'power-elements' ];
		}
		
		
		/**
		 * Register social widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _register_controls() {
			$this->start_controls_section(
				'section_logo_content',
				[
					'label' => __( 'Content', 'power-elements' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'peeao_logo_use_custom',
				[
					'label' => esc_html__( 'Use custom logo?', 'power-elements' ),
					'type' => Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Yes', 'power-elements' ),
					'label_off' => esc_html__( 'No', 'power-elements' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);
			$this->add_control(
				'peeao_logo_image',
				[
					'label'   => __( 'Choose Logo', 'power-elements' ),
					'type'    => Controls_Manager::MEDIA,
					'default' => [
						'url' => \Elementor\Utils::get_placeholder_image_src(),
					],
					'condition' => [
						'peeao_logo_use_custom' => 'yes',
					]
				]
			);
			
			$this->add_group_control(
				Group_Control_Image_Size::get_type(),
				[
					'name'      => 'peeao_logo_thumbnail',
					'default'   => 'full',
					'separator' => 'none',
				]
			);
			
			$this->end_controls_section();

		}
			
		/**
		 * Render social widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function render() {
			$settings   = $this->get_settings();
			$custom_logo_id = get_theme_mod( 'custom_logo' );
			$custom_logo = wp_get_attachment_image_src( $custom_logo_id , 'full' );

			$image_html = ($settings['peeao_logo_use_custom'] == 'yes') ? 
							Group_Control_Image_Size::get_attachment_image_html( $settings, 'peeao_logo_thumbnail', 'peeao_logo_image' ) :
							'<img src="'.esc_url($custom_logo[0]).'" alt="'.get_bloginfo().' - '.esc_html__('Logo', 'power-elements').'">';
			
        
			echo '<div class="peeao-wrapper">

				<div class="logo">
					<a href="'.home_url('/').'">' . $image_html . '</a>
				</div>
			
			</div>';
		}
		
		/**
		 * Render social widget output in the editor.
		 *
		 * Written as a Backbone JavaScript template and used to generate the live preview.
		 *
		 * @since 1.0.0
		 * @access protected
		 */
		protected function _content_template() {}
	}