<?php

namespace PowerElements;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Icons_Manager;



if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Power Element button widget.
 *
 * Elementor widget that displays a button with the ability to control every
 * aspect of the button design.
 *
 * @since 1.0.0
 */
class PowerElements_Button extends Widget_Base
{

	/**
	 * Get widget name.
	 *
	 * Retrieve button widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'power_elements_button';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve button widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return __('Button', 'power-elements');
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve button widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-button';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the button widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['power-elements'];
	}



	/**
	 * Register button widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls()
	{
		$this->start_controls_section(
			'peeao_section_button',
			[
				'label' => __('Button', 'power-elements'),
			]
		);

		$this->add_control(
			'peeao_button_style',
			[
				'label'        => __('Type', 'power-elements'),
				'type'         => Controls_Manager::SELECT,
				'default'      => 'effect_1',
				'options'      => [
					'effect_1'  => __('Effect 1', 'power-elements'),
					'effect_2'  => __('Effect 2', 'power-elements'),
					'effect_3'  => __('Effect 3', 'power-elements'),
					'effect_4'  => __('Effect 4', 'power-elements'),
					'effect_5'  => __('Effect 5', 'power-elements'),
					'effect_6'  => __('Effect 6', 'power-elements'),
					'effect_7'  => __('Effect 7', 'power-elements'),
					'effect_8'  => __('Effect 8', 'power-elements'),
					'effect_9'  => __('Effect 9', 'power-elements'),
					'effect_10' => __('Effect 10', 'power-elements'),
					'effect_11' => __('Effect 11', 'power-elements'),
					'effect_12' => __('Effect 12', 'power-elements'),
					'effect_13' => __('Effect 13', 'power-elements'),
				]
			]
		);

		$this->add_control(
			'peeao_button_text',
			[
				'label'       => __('Text', 'power-elements'),
				'type'        => Controls_Manager::TEXT,
				'default'     => __('Click here', 'power-elements'),
				'placeholder' => __('Click here', 'power-elements'),
			]
		);

		$this->add_control(
			'peeao_button_link',
			[
				'label'       => __('Link', 'power-elements'),
				'type'        => Controls_Manager::URL,
				'placeholder' => __('https://your-link.com', 'power-elements'),
				'default'     => [
					'url' => '#',
				],
			]
		);

		$this->add_responsive_control(
			'peeao_button_align',
			[
				'label'        => __('Alignment', 'power-elements'),
				'type'         => Controls_Manager::CHOOSE,
				'options'      => [
					'left'    => [
						'title' => __('Left', 'power-elements'),
						'icon'  => 'fa fa-align-left',
					],
					'center'  => [
						'title' => __('Center', 'power-elements'),
						'icon'  => 'fa fa-align-center',
					],
					'right'   => [
						'title' => __('Right', 'power-elements'),
						'icon'  => 'fa fa-align-right',
					],
					'justify' => [
						'title' => __('Justified', 'power-elements'),
						'icon'  => 'fa fa-align-justify',
					],
				],
				'prefix_class' => 'elementor%s-align-',
				'default'      => '',
			]
		);


		$this->add_control(
			'peeao_button_icon',
			[
				'label' => esc_html__('Icon', 'power-elements'),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => '',
					'library' => 'peeaoicons',
				],
			]
		);
		$this->add_control(
			'peeao_button_icon_align',
			[
				'label'     => __('Icon Position', 'power-elements'),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'power_button_icon_left',
				'options'   => [
					'power_button_icon_right' => __('Right', 'power-elements'),
					'power_button_icon_left'  => __('Left', 'power-elements'),
				],
				'condition' => [
					'peeao_button_icon!' => '',
				],
			]
		);

		$this->add_responsive_control(
			'peeao_button_icon_indent',
			[
				'label'     => __('Icon Spacing', 'power-elements'),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'max' => 50,
					],
				],
				'condition' => [
					'peeao_button_icon!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .power_button_icon_right i' => 'margin-left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .power_button_icon_left i'  => 'margin-right: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'peeao_button_view',
			[
				'label'   => __('View', 'power-elements'),
				'type'    => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'peeao_button_section_style',
			[
				'label' => __('Button', 'power-elements'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'peeao_button_content_typography',
				'label'    => __('Typography', 'power-elements'),
				'selector' => '{{WRAPPER}} .power_button',
			]
		);
		$this->add_control(
			'peeao_button_before_animation',
			[
				'label' => __('Hover Animation', 'power-elements'),
				'type'  => Controls_Manager::HOVER_ANIMATION,
			]
		);
		$this->start_controls_tabs('peeao_button_tabs_button_style');

		$this->start_controls_tab(
			'peeao_button_tab_style_normal',
			[
				'label' => __('Normal', 'power-elements'),
			]
		);

		$this->add_control(
			'peeao_button_ext_color',
			[
				'label'     => __('Text Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#0071f0',
				'selectors' => [
					'{{WRAPPER}} .button_value, {{WRAPPER}} .button_value' => 'color: {{VALUE}};',
					'{{WRAPPER}} .power_button-link' => 'text-decoration-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'peeao_button_background_color',
				'label'    => __('Background Color', 'power-elements'),
				'types'    => ['classic', 'gradient'],
				'selector' => '{{WRAPPER}} .power_button',
			]
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'peeao_button_border',
				'fields_options' => [
					'border' => [
						'default' => 'none',
					],
					'width' => [
						'default' => [
							'top' => '0',
							'right' => '0',
							'bottom' => '0',
							'left' => '0',
							'isLinked' => false,
						],
					],
					'color' => [
						'default' => '',
					],
				],
				'selector'  => '{{WRAPPER}} a.power_button, {{WRAPPER}} .power_button',
			]
		);

		$this->add_responsive_control(
			'peeao_button_border_radius',
			[
				'label'      => __('Border Radius', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'default'    => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
					'unit' => 'px',
					'isLinked' => '',
				],
				'selectors'  => [
					'{{WRAPPER}} a.power_button, {{WRAPPER}} .power_button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'peeao_button_box_shadow',
				'selector' => '{{WRAPPER}} .power_button',
			]
		);

		$this->add_responsive_control(
			'peeao_button_text_padding',
			[
				'label'      => __('Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} a.power_button, {{WRAPPER}} .power_button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'peeao_button_icon_color',
			[
				'label'     => __('Icon Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} a.power_button .button_value .power_button_icon' => 'color: {{VALUE}};',
					'{{WRAPPER}} .power_button .button_value .power_button_icon' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_tab();

		$this->start_controls_tab(
			'peeao_button_tab_style_hover',
			[
				'label' => __('Hover', 'power-elements'),
			]
		);

		$this->add_control(
			'peeao_button_hover_text_color',
			[
				'label'     => __('Text Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.power_button:hover .button_value, {{WRAPPER}} .power_button:hover .button_value, {{WRAPPER}} a.power_button:focus, {{WRAPPER}} .power_button:focus' => 'color: {{VALUE}};',
					'{{WRAPPER}} .power_button:hover .power_button-link' => 'text-decoration-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'peeao_button_background_hover_color',
				'label'    => __('Background Color', 'power-elements'),
				'types'    => ['classic', 'gradient'],
				'selector' => '{{WRAPPER}} .power_button:hover',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'peeao_button_border_hover',
				'selector'  => '{{WRAPPER}} .power_button:hover',
			]
		);

		$this->add_responsive_control(
			'peeao_button_hover_border_radius',
			[
				'label'      => __('Border Radius', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'default'    => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
					'unit' => 'px',
					'isLinked' => '',
				],
				'selectors'  => [
					'{{WRAPPER}} a.power_button:hover, {{WRAPPER}} .power_button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'peeao_button_hover_box_shadow',
				'selector' => '{{WRAPPER}} .power_button:hover',
			]
		);

		$this->add_responsive_control(
			'peeao_button_hover_text_padding',
			[
				'label'      => __('Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} a.power_button:hover, {{WRAPPER}} .power_button:hover' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]

		);
		$this->add_control(
			'peeao_button_hover_icon_hover_color',
			[
				'label'     => __('Icon Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.power_button:hover .button_value .power_button_icon' => 'color: {{VALUE}};',
					'{{WRAPPER}} .power_button:hover .button_value .power_button_icon' => 'color: {{VALUE}};',
					'{{WRAPPER}} a.power_button:focus .power_button_icon' => 'color: {{VALUE}};',
					'{{WRAPPER}} .power_button:focus .power_button_icon' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();


		$this->start_controls_section(
			'peeao_button_section_effect_style',
			[
				'label' => __('Effect', 'power-elements'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'peeao_button_section_effect_use',
			[
				'label'        => __('Use Effect?', 'power-elements'),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => __('Icon', 'power-elements'),
				'label_off'    => __('Image', 'power-elements'),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);
		$this->start_controls_tabs('peeao_button_tabs_style_after');
		$this->start_controls_tab(
			'peeao_button_tab_style_after',
			[
				'label' => __('Normal', 'power-elements'),
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);
		$this->add_control(
			'peeao_button_background_color_after',
			[
				'label'     => __('Layer Background Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.power_button:after, {{WRAPPER}} .power_button:after' => 'background-color: {{VALUE}};',

				],
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'peeao_button_border_after',
				'selector'  => '{{WRAPPER}} .power_button:after',
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_button_after_border_radius',
			[
				'label'      => __('Border Radius', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'default'    => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
					'unit' => 'px',
					'isLinked' => '',
				],
				'selectors'  => [
					'{{WRAPPER}} a.power_button:after, {{WRAPPER}} .power_button:after' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'peeao_button_after_box_shadow',
				'selector' => '{{WRAPPER}} .power_button:after',
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'peeao_button_after_text_padding',
			[
				'label'      => __('Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} a.power_button:after, {{WRAPPER}} .power_button:after' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
				'separator'  => 'before',
			]

		);
		$this->end_controls_tab();

		$this->start_controls_tab(
			'peeao_button_tab_style_before',
			[
				'label' => __('Hover', 'power-elements'),
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);
		$this->add_control(
			'peeao_button_background_before_color',
			[
				'label'     => __('Seceondary Background Color', 'power-elements'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.power_button:before, {{WRAPPER}} .power_button:before' => 'background-color: {{VALUE}};',
				],
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'peeao_button_border_before',
				'selector'  => '{{WRAPPER}} .power_button:before',
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);
		$this->add_responsive_control(
			'peeao_button_before_border_radius',
			[
				'label'      => __('Border Radius', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'default'    => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
					'unit' => 'px',
					'isLinked' => 'true',
				],
				'selectors'  => [
					'{{WRAPPER}} a.power_button, {{WRAPPER}} .power_button:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'peeao_button_before_box_shadow',
				'selector' => '{{WRAPPER}} .power_button:before',
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'peeao_button_before_text_padding',
			[
				'label'      => __('Padding', 'power-elements'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} a.power_button:before, {{WRAPPER}} a.power_button:before, {{WRAPPER}} .power_button:before, {{WRAPPER}} .power_button:before' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'   => [
					'peeao_button_section_effect_use' => 'yes',
				],
				'separator'  => 'before',
			]

		);
		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();
	}



	/**
	 * Render button widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected 
	 */
	protected function render()
	{
		$settings = $this->get_settings();

		$this->add_render_attribute('wrapper', 'class', 'power_button-wrapper');

		if (!empty($settings['peeao_button_link']['url'])) {
			$this->add_render_attribute('button', 'href', $settings['peeao_button_link']['url']);
			$this->add_render_attribute('button', 'class', 'power_button-link');

			if ($settings['peeao_button_link']['is_external']) {
				$this->add_render_attribute('button', 'target', '_blank');
			}

			if ($settings['peeao_button_link']['nofollow']) {
				$this->add_render_attribute('button', 'rel', 'nofollow');
			}
		}
		if($settings['peeao_button_section_effect_use']){
			$effect = 'btn_use_effect ' . $settings['peeao_button_style'];
		}else{
			$effect = '';
		}
		$this->add_render_attribute('button', 'class', [
			'power_button',
			$effect,
			$settings['peeao_button_icon_align']
		]);
		$this->add_render_attribute('button', 'role', 'button');
		if ($settings['peeao_button_before_animation']) {
			$this->add_render_attribute('button', 'class', 'elementor-animation-' . $settings['peeao_button_before_animation']);
		}

		$this->add_render_attribute([
			'content-wrapper' => [
				'class' => 'power_button-content-wrapper',
			],
			'text'            => [
				'class' => ['power_button-text', 'button_value']
			],
		]);

		$this->add_inline_editing_attributes('text', 'none');  ?>

		<div class="peeao-wrapper">
			<div <?php echo $this->get_render_attribute_string('wrapper'); ?>>
				<a <?php echo $this->get_render_attribute_string('button'); ?>>
					<?php if (!empty($settings['peeao_button_icon']['value'])) : ?>
						<?php if ($settings['peeao_button_icon_align'] == 'power_button_icon_right') : ?>
							<span <?php echo $this->get_render_attribute_string('text'); ?>><?php echo $settings['peeao_button_text']; ?><?php Icons_Manager::render_icon($settings['peeao_button_icon'], ['class'    => 'power_button_icon']); ?></span>
						<?php else : ?>
							<span <?php echo $this->get_render_attribute_string('text'); ?>><?php Icons_Manager::render_icon($settings['peeao_button_icon'], ['class'    => 'power_button_icon']); ?><?php echo $settings['peeao_button_text']; ?></span>
						<?php endif; ?>
					<?php else : ?> 
						<span <?php echo $this->get_render_attribute_string('text'); ?>><?php echo $settings['peeao_button_text']; ?></span>
					<?php endif; ?>
				</a>
			</div>
		</div>
<?php
	}

	/**
	 * Render button widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _content_template()
	{
	}
}
